/**
 * A Java interface for the Updatable Priority Queue ADT, which allows the user
 * to query and change the priority of items in the queue.
 * 
 * Each implementation of this interface shall define a special "sentinel"
 * priority value that indicates an item is not in the queue.  Setting an
 * item's priority to this value in fact removes it from the queue.  Removing a
 * non-member item is a no-op.
 * 
 * In addition, adding an item may be accomplished, from the user side, simply
 * by setting its priority to a non-sentinel value.
 * 
 * @param <T> the type of the items stored in the queue.
 * @param <P> the type of the priority of each item.
 * @author Jadrian Miles
 */
public interface UpdatablePriorityQueue<T, P extends Comparable<? super P>>
        extends PriorityQueue<PriorityPair<T, P>> {
    /**
     * Adds an item, with the priority given in the pair, to the queue.  This is
     * equivalent to setPriority(itemPair.item(), itemPair.priority()), and
     * therefore may, depending on the priority and the current state of the
     * queue, add, remove, or modify the priority of the item.
     */
    public void add(PriorityPair<T, P> itemPair);
    
    /**
     * Adds an item, with the given priority, to the queue.  This is equivalent
     * to setPriority(item, priority), and therefore may, depending on the
     * priority and the current state of the queue, add, remove, or modify the
     * priority of the item.
     */
    public void add(T item, P priority);
    
    /**
     * Sets the priority of a given item.  The consequences of this operation
     * vary depending on two conditions:
     *    OLD: the item is already in the queue.
     *    DEL: "newPriority" is the non-member priority sentinel value.
     * The four cases covered by combinations of these conditions are:
     *  1. If  OLD and !DEL, this method changes the item's priority.
     *  2. If !OLD and !DEL, this method adds the item.
     *  3. If  OLD and  DEL, this method removes the item.
     *  4. If !OLD and  DEL, this method is a no-op (removing a non-member).
     * The return value is !DEL (true in cases 1 and 2; false in 3 and 4).
     * @return true if the pair (item, newPriority) is in the queue after the
     *    call.
     */
    public boolean setPriority(T item, P newPriority);
    
    /**
     * Returns the priority of the given item, or, if that item is not in the
     * queue, an implementation-specific sentinel value.
     */
    public P getPriority(T item);
    
    /**
     * Returns the sentinel value used by getPriority() to indicate items not
     * currently stored in the queue.
     */
    public P nonMemberPriority();
}
